/* Copyright (C) 2003-2004 by Peter Eastman

  Modified by Jeremiah Moss

   This program is free software; you can redistribute it and/or modify it under the
   terms of the GNU General Public License as published by the Free Software
   Foundation; either version 2 of the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful, but WITHOUT ANY 
   WARRANTY; without even the implied warranty of MERCHANTABILITY or FITNESS FOR A 
   PARTICULAR PURPOSE.  See the GNU General Public License for more details. */

package artofillusion.image.filter;

import artofillusion.*;
import artofillusion.image.*;
import artofillusion.math.*;
import artofillusion.object.*;
import artofillusion.ui.*;
import java.io.*;
import buoy.widget.*;
import java.util.concurrent.*;

/** This is an image filter which blurs an image. */

public class FastBlurFilter extends ImageFilter
{

  public FastBlurFilter()
  {
    myThreadStates = new int[Runtime.getRuntime().availableProcessors()];
    for(int x=0; x < Runtime.getRuntime().availableProcessors(); x++)
    {
      FastBlurFilter.myThreadStates[x] = 0;
    }
  }

  
  
  /** Get the name of this filter.*/

  public String getName()
  {
    return Translate.text("Fast Blur");
  }
  
  /** Apply the filter to an image.
      @param image      the image to filter
      @param scene      the Scene which was rendered to create the image
      @param camera     the camera from which the Scene was rendered
      @param cameraPos  the position of the camera in the scene
  */
  
  public void filterImage(ComplexImage image, Scene scene, SceneCamera camera, CoordinateSystem cameraPos)
  {
    int numCores = Runtime.getRuntime().availableProcessors();
    
    int radius = (int) (0.5f*paramValue[0]*image.getHeight());
    if (radius < 1)
      return;
    float mask[] = createMask(radius);

	BFrame f = new BFrame("Blur Progress");
	BorderContainer bc = new BorderContainer();
	f.setContent(bc);

	BProgressBar bloomProgress = new BProgressBar(BProgressBar.HORIZONTAL, 0, 100);
	BLabel bloomStage = new BLabel("Red 1/3");

	bloomProgress.setValue(0);
	bloomProgress.setProgressText("0%");
	bloomProgress.setShowProgressText(true);

	bc.add(bloomProgress, BorderContainer.CENTER);
	bc.add(bloomStage, BorderContainer.NORTH);

	f.pack();
    f.setVisible(true);
    f.toFront();

    if(numCores == 1)
    {
      // Single core: Don't need the overhead of threading
      filterComponent(image, ComplexImage.RED, radius, mask, bloomProgress);
      bloomStage.setText("Green 2/3");
      filterComponent(image, ComplexImage.GREEN, radius, mask, bloomProgress);
      bloomStage.setText("Blue 3/3");
      filterComponent(image, ComplexImage.BLUE, radius, mask, bloomProgress);
    } else
    {
      // Multiple cores
      tFilter(image, ComplexImage.RED, radius, mask, bloomProgress);
      bloomStage.setText("Green 2/3");
      tFilter(image, ComplexImage.GREEN, radius, mask, bloomProgress);
      bloomStage.setText("Blue 3/3");
      tFilter(image, ComplexImage.BLUE, radius, mask, bloomProgress);
    }
	f.setVisible(false);
	f.dispose();
  }
  
  /** Apply the filter to one component of an image. */
  /* Rewritten with a two-pass blur */  
  private void filterComponent(ComplexImage image, int component, int radius, float mask[], BProgressBar progress)
  {
    Thread currentThread = Thread.currentThread();
    int maskWidth = 2*radius+1;
    int width = image.getWidth(), height = image.getHeight();
    float baseImage[] = new float [width*height];
    float blur[] = new float[width*height];
    float blur2[] = new float[width*height];
    
    if((blur == null) || (blur2 == null))
    {
      throw new NullPointerException("Not enough memory for Fast Blur filter.");
    }
    
    progress.setValue(0);
    progress.setProgressText("0%");

    for (int i = 0; i < width; i++)
    {
      if (currentThread.isInterrupted())
	return;
      for (int j = 0; j < height; j++)
      {
	baseImage[i+j*width] = image.getPixelComponent(i, j, component);
      }
    }


    /* First pass */

    float value = 0;

    for (int i = 0; i < width; i++)
    {
	  progress.setValue((int)(50*((float)i/(float)width)));
	  progress.setProgressText(Integer.toString(progress.getValue()) + "% " + Integer.toString(i) + "/" + Integer.toString(width));
      if (currentThread.isInterrupted())
        return;
      for (int j = 0; j < height; j++)
      {

        int basex = i, basey = j-radius;
        int ystart = (basey < 0 ? -basey : 0);
        int yend = (basey+maskWidth >= height ? height-basey : maskWidth);
        for (int y = ystart; y < yend; y++)
        {
	  value = baseImage[basex+(basey+y)*width];
          int maskBase = y;//y*maskWidth;
          int imageBase = basex+(j)*width;
          blur[imageBase] += mask[maskBase]*value;
        }
      }
    }


    /* Second pass */
    for (int i = 0; i < width; i++)
    {
	  progress.setValue((int)(50+50*((float)i/(float)width)));
	  progress.setProgressText(Integer.toString(progress.getValue()) + "% " + Integer.toString(i) + "/" + Integer.toString(width));
      if (currentThread.isInterrupted())
        return;
      for (int j = 0; j < height; j++)
      {
        int basex = i-radius, basey = j;
        int xstart = (basex < 0 ? -basex : 0);
        int xend = (basex+maskWidth >= width ? width-basex : maskWidth);
        for (int x = xstart; x < xend; x++)
        {
	  value = blur[basex+x+(basey)*width];
	  int maskBase = x;
          int imageBase = i+(basey)*width;
            blur2[imageBase] += mask[maskBase]*value;
        }
      }
    }

    progress.setValue(100);
    progress.setProgressText("100%");

    image.setComponentValues(component, blur2);
  }
  
  /** Build the mask. */
  
  private float [] createMask(int radius)
  {
    int size = 2*radius+1;
    float mask[] = new float [size];
    for (int i = 0; i < radius; i++)
    {
//	double sd = 0.5;
          float dist2 = i*i;
          float d = (dist2/(float) (radius*radius));
          float value = d*(d-2.0f)+1.0f;

//	float value = (float)((1/(Math.sqrt(2.0*Math.PI)*sd*sd))*Math.pow(Math.E,(-(Math.pow(dist2,2.0))/(2*Math.pow(Math.E,2.0)))));
          mask[radius-i] = value;
          mask[radius+i] = value;
    }

    // Normalize the mask.
    
    double sum = 0.0;
    for (int i = 0; i < mask.length; i++)
      sum += mask[i];
    float scale = (float) (1.0/sum);
    for (int i = 0; i < mask.length; i++)
      mask[i] *= scale;
    return mask;
  }
  
static volatile int[] myThreadStates;
  public void tFilter(ComplexImage image, int component, int radius, float mask[], BProgressBar progress)
  {
    
    // 0 = no thread, or thread is sleeping
    // 2 = Started Thread
    // 4 = Finished Thread
    int numThreads = Runtime.getRuntime().availableProcessors();
    Thread[] myThreads = new Thread[numThreads];
    Thread currentThread = Thread.currentThread();
    float[][] newImages = new float[numThreads][];
    
    LinkedBlockingQueue<Integer> lineList = new LinkedBlockingQueue<Integer>();
    
    int start;
    int end;

    int numLines = image.getWidth();

    for(int x=0; x < image.getWidth(); x++)
    {
        lineList.add(x);
    }
    
    for(int x=0; x < numThreads; x++)
    {
      FastBlurFilter.myThreadStates[x] = 0;
    }
    
    for(int x=0; x < numThreads; x++)
    {
      start = (int)((float)x/numThreads*numLines);
      end = (int)(((float)x+1)/numThreads*numLines);
      newImages[x] = new float[image.getWidth()*image.getHeight()];
      ThreadedBlur newFilter = new ThreadedBlur(image, component, radius, mask, progress, start, end, x, newImages[x]);
      myThreads[x] = new Thread(newFilter);
      myThreads[x].start();
    }
    
    boolean allFinished = true;
    do
    {
      allFinished = true;
      try
      {
	currentThread.sleep(100);
      } catch(java.lang.InterruptedException E)
      {
	System.out.println("Blur Filter interrupted.");
      }
      for(int x=0; x < numThreads; x++)
      {
	if(FastBlurFilter.myThreadStates[x] != 4)
	{
	  allFinished = false;
	}
      }
    } while(allFinished == false);
    
    float[] imageC = new float[image.getWidth() * image.getHeight()];
    // Piece image back together
    for(int x=0; x < numThreads; x++)
    {
      start = (int)((float)x/numThreads*numLines);
      end = (int)(((float)x+1)/numThreads*numLines);
      for (int i = start; i < end; i++)
      {
      if (currentThread.isInterrupted())
	return;
      	for (int j = 0; j < image.getHeight(); j++)
	{
	  imageC[i+j*image.getWidth()] += newImages[x][i+j*image.getWidth()];
	}
      }
    }
    image.setComponentValues(component, imageC);
  }
  


  /** Get a list of parameters which affect the behavior of the filter. */
  
  public TextureParameter [] getParameters()
  {
    return new TextureParameter [] {new TextureParameter(this, Translate.text("Radius"), 0.0, 1.0, 0.05)};
  }

  /** Write a serialized description of this filter to a stream. */
  
  public void writeToStream(DataOutputStream out, Scene theScene) throws IOException
  {
    out.writeDouble(paramValue[0]);
  }

  /** Reconstruct this filter from its serialized representation. */
  
  public void initFromStream(DataInputStream in, Scene theScene) throws IOException
  {
    paramValue[0] = in.readDouble();
  }
}
