-- Special error code. This file is vitally important for proper error handling.
-- It is loaded directly, and it supplies a way to get a Lua stack trace
-- for C#.

-- We can't rely on custom print handlers at this point: Make sure we are using
-- the native print.

-- Load required C# functions into local scope.
local CPrint = _CSHARP.CPrint;
local _FORCEQUIT = _CSHARP._FORCEQUIT;
local _CERROR = _CSHARP._CERROR;

CPrint(_VERSION);

CPrint("Error handling loading");


-- Override error function to get more info before falling out to C#.
local oldError = error;


local function luaError(...)
	CPrint("====LUA====")
	CPrint(...);
	CPrint("===STACK===")
	CPrint(debug.traceback());
	CPrint("==END LUA==")
	_FORCEQUIT();
end

local function CError(...)
	CPrint("=LUA STACK=")
	CPrint(debug.traceback());
	CPrint("==END LUA==")
	_CERROR(...)
end

error = luaError;

function doFile(name)
	CPrint("loading "..name);
	file,code = loadfile(name)
	if file==nil then
	   CPrint("Error loading "..name..".")
	   luaError(code)
	else
	   xpcall(file, function(err)
		   if type(err)=="userdata" then
			   CPrint("Error in C# code.")
			   CError(err)
		   else
			   CPrint("Error: "..err)
			   luaError(err)
		   end
	   end)
	end
	CPrint("loaded "..name);
end

function doString(text)
	CPrint("running string: "..text);

	string,code = loadstring(text)
	if string==nil then
	   CPrint("Error in string: "..text..".")
	   luaError(code)
	else
	   xpcall(string, function(err)
		   if type(err)=="userdata" then
			   CPrint("Error in C# code.")
			   CError(err)
		   else
			   CPrint("Error: "..err)
			   luaError(err)
		   end
	   end)
	end
	CPrint("ran string: "..text);
end

function printTable(table)
	for k,v in pairs(table) do
		if(type(v) == "table" or type(v) == "function" or type(v) == "userdata") then
			CPrint("[" .. k .. "] type: " .. type(v));
		else
			CPrint("[" .. k .. "]=" .. v);
		end	
	end
end

--------------------------------------------------------------------------------
-- Before this line, use CPrint to print directly to C#.
--------------------------------------------------------------------------------

local oldPrint = print;

function print(...)
	toPrint = {...}
	printString = ""
	for k, v in pairs(toPrint) do
		if(type(v)=="table") then
			CPrint("(table) " .. k)
			printTable(v);
		elseif type(v)=="boolean" then
			if v then
				printString = printString .. "true"
			else
				printString = printString .. "false"
			end
		elseif type(v)=="userdata" then
			printString = "variable "..k.." is userdata."
		else
			printString = printString .. v
		end
	end
	CPrint(printString);
end


print("Error handling loaded");

errorLoaded = true;

-- Wipe these functions from global scope - not even base functions can access
-- them directly. I want to make absolutely sure that error handling stays here. 
_CSHARP.CPrint = nil;
_CSHARP._FORCEQUIT = nil;
_CSHARP._CERROR = nil;

